package com.itextpdf.text.pdf.security;

import java.security.GeneralSecurityException;
import java.security.PrivateKey;
import java.security.Signature;
import java.io.FileWriter;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.Date;

/**
 * Fixed version of PrivateKeySignature that ignores the Apple provider
 * and always uses the default provider on macOS.
 */
public class PrivateKeySignature implements ExternalSignature {
    
    private PrivateKey pk;
    private String hashAlgorithm;
    private String encryptionAlgorithm;
    
    private void log(String message) {
        try {
            FileWriter writer = new FileWriter("/tmp/log.txt", true);
            String timestamp = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss.SSS").format(new Date());
            writer.write("[" + timestamp + "] IZENPE FIX: " + message + "\n");
            writer.flush();
            writer.close();
        } catch (IOException e) {
            // Ignore logging errors
        }
    }
    
    public PrivateKeySignature(PrivateKey pk, String hashAlgorithm, String provider) {
        this.pk = pk;
        this.hashAlgorithm = DigestAlgorithms.getDigest(DigestAlgorithms.getAllowedDigests(hashAlgorithm));
        this.encryptionAlgorithm = pk.getAlgorithm();
        
        if (this.encryptionAlgorithm.startsWith("EC")) {
            this.encryptionAlgorithm = "ECDSA";
        }
        
        // Log constructor call
        log("PrivateKeySignature constructor called");
        log("  Original provider: " + provider);
        log("  Hash algorithm: " + hashAlgorithm);
        log("  Encryption algorithm: " + encryptionAlgorithm);
        log("  Using default provider instead of Apple");
        
        // Debug output
        System.out.println("🔧 IZENPE FIX: PrivateKeySignature created");
        System.out.println("   Original provider: " + provider);
        System.out.println("   Using default provider instead");
    }
    
    public String getHashAlgorithm() {
        return hashAlgorithm;
    }
    
    public String getEncryptionAlgorithm() {
        return encryptionAlgorithm;
    }
    
    public byte[] sign(byte[] message) throws GeneralSecurityException {
        String algorithm = hashAlgorithm + "with" + encryptionAlgorithm;
        
        // Log signing attempt
        log("sign() method called with algorithm: " + algorithm);
        log("  Message length: " + message.length + " bytes");
        
        try {
            // Always use default provider (ignore Apple provider)
            Signature sig = Signature.getInstance(algorithm);
            sig.initSign(pk);
            sig.update(message);
            
            byte[] signature = sig.sign();
            
            // Log successful signing
            log("  Signature created successfully, length: " + signature.length + " bytes");
            log("  Used default provider instead of Apple");
            
            System.out.println("🔧 IZENPE FIX: Signed with algorithm " + algorithm + " using default provider");
            return signature;
        } catch (Exception e) {
            // Log signing failure
            log("  Signing failed: " + e.getClass().getSimpleName() + ": " + e.getMessage());
            throw e;
        }
    }
}
